import React, { Component } from "react";
import {
    View,
    Text,
    Image,
    StyleSheet,
    TextInput,
    TouchableOpacity,
    Linking,
    Platform
} from "react-native";
import { connect } from "react-redux";
import { WithFetching } from "../../HOC";
import { UIFont, fonts, fontSize, icons } from "../../assets";
import { FooterDetail } from "../../components";
import { API_CONTACT_EMAIL, API_SENT_EMAIL } from "../../services/WebService";
import { DUMP } from "../../actions/ActionTypes";
import { generalAction } from "../../actions/GeneralActions";
import Utils from "../../Utils";
import { pop } from "../../services/NavigationService";

class ContactUs extends Component {
    state = {
        message: "",
        contact_number: "",
        email_address: ""
    };

    componentDidMount() {
        this.fetchContactData();
    }

    fetchContactData = () => {
        this.props.generalAction(
            API_CONTACT_EMAIL,
            {},
            DUMP.SUCCESS,
            DUMP.FAILURE,
            ({ contact_number, email_address }) => {
                this.setState({ contact_number, email_address });
            },
            () => {
                this.fetchContactData();
            },
            true
        );
    };

    onChangeText = message => {
        this.setState({ message });
    };

    footer() {
        return (
            <FooterDetail
                style={{ position: "absolute", bottom: 0 }}
                text={"SUBMIT"}
                backgroundColor={{ backgroundColor: "#08548c" }}
                textColor={{ color: "#ffffff" }}
                clickBtn={this.sendMessage}
            />
        );
    }

    sendMessage = () => {
        let payload = new FormData();
        payload.append("user_id", this.props.user.data.id);
        payload.append("message", this.state.message);
        this.props.generalAction(
            API_SENT_EMAIL,
            payload,
            DUMP.SUCCESS,
            DUMP.FAILURE,
            this.emailSendSuccessfully,
            this.emailSendFailure
        );
        this.props.cbShowLoader(true);
    };

    emailSendSuccessfully = user => {
        Utils.showMessage(
            "Success",
            "Your email has been sent successfully",
            "success"
        );
        pop();
        this.props.cbShowLoader(false);
    };

    emailSendFailure = err => {
        this.props.cbShowLoader(false);
        Utils.showError(err);
    };

    openLink = (url, err) => {
        Linking.canOpenURL(url)
            .then(supported => {
                if (supported) {
                    Linking.openURL(url);
                } else {
                    this.showMessage("Error", err);
                }
            })
            .catch(err => {
                this.showMessage("Error", err);
            });
    };

    render() {
        const { contact_number, email_address, message } = this.state;
        return (
            <View style={styles.container}>
                <TouchableOpacity
                    style={styles.eachOne}
                    onPress={() =>
                        this.openLink(`tel:${""}`, "Can't open dialer")
                    }
                >
                    <Image
                        style={styles.icon}
                        resizeMode="contain"
                        source={icons.call_contact}
                    />
                    <Text style={styles.eachText}>{contact_number}</Text>
                </TouchableOpacity>
                <TouchableOpacity style={styles.eachOne}>
                    <Image
                        style={styles.icon}
                        resizeMode="contain"
                        source={icons.email_contact}
                    />
                    <Text style={styles.eachText}>{email_address}</Text>
                </TouchableOpacity>
                <View style={{ flex: 5, paddingHorizontal: 20 }}>
                    <TextInput
                        numberOfLines={10}
                        multiline={true}
                        onChangeText={this.onChangeText}
                        value={message}
                        style={styles.inputMessageStyle}
                        placeholder="Please enter your Feedback"
                    />
                </View>
                {this.footer()}
            </View>
        );
    }
}

const actions = { generalAction };

const mapStateToProps = state => ({
    user: state.user
});

export default connect(mapStateToProps, actions)(WithFetching(ContactUs));

const styles = StyleSheet.create({
    container: {
        flex: 1,
        backgroundColor: "#f7f7f7"
    },
    eachOne: {
        flex: 1,
        paddingHorizontal: 20,
        paddingTop: 20,
        paddingBottom: 10,
        flexDirection: "row"
    },
    eachText: {
        ...UIFont.font(fonts.regular, 16, "#525252"),
        flex: 1,
        alignSelf: "center",
        paddingLeft: 10
    },
    icon: {
        margin: 10,
        paddingLeft: 15,
        alignSelf: "center"
    },
    inputMessageStyle: {
        marginVertical: 10,
        borderWidth: 1,
        borderColor: "#dde3f6",
        borderRadius: 3,
        height: Platform.OS && 120,
        textAlignVertical: "top",
        width: "100%"
    }
});
