import React, { Component, Fragment } from "react";
import { connect } from "react-redux";
import { View, Text, TouchableOpacity, Image } from "react-native";
import { WithFetching } from "../../HOC";
import { compose } from "redux";
import { Images, icons, colors } from "../../assets";
import { pop, push, popToTop } from "../../services/NavigationService";
import MediaPicker from "../../services/MediaPicker";
import {
    IMAGE_BASE_URL,
    API_GET_COMMENTS_LIST,
    API_GET_CONTRACTOR
} from "../../services/WebService";
import { AppButton,Supporticon } from "../../components";
import Contractor from "./Contractor";
import Facility from "./Facility";
import Utils from "../../Utils";
import { USER, DUMP } from "../../actions/ActionTypes";
import { generalAction, generalSaveAction } from "../../actions/GeneralActions";
import styles from "./styles";

class Profile extends Component {
    constructor(props, context) {
        super(props, context);

        this.state = {
            image_url: props.user.data.image_url || Images.cardImage,
            editable: true,
            isDpEdited: false,
            commentsList: []
        };


    }

    componentDidMount() {
        this.props.navigation.setParams({
            onEdit: () => this.onEdit()
        });
        this.fetchData(false, 1, API_GET_COMMENTS_LIST, false);
        this.fetchData(false, 1, API_GET_CONTRACTOR, true);
    }

    fetchData = (
        isConcat: boolean = false,
        page: Number = 1,
        api,
        updateAlways
    ) => {
        this.setState({ isFetching: true });
        this.props.generalAction(
            api,
            {
                user_id: this.props.user.data.id,
                facility_id: 0
            },
            updateAlways ? USER.SUCCESS : DUMP.SUCCESS,
            updateAlways ? USER.FAILURE : DUMP.FAILURE,
            data => {


                !updateAlways && this.setState({ commentsList: data });
                updateAlways && this.onUpdateSuccess(data, true);
            },
            () => {
                this.setState({ isFetching: false });
            },
            true,
            isConcat
        );

    };

    onEdit = () => {
        this.setState({ editable: true });
    };

    onImagePicked = (uri, imageObj) => {
        this.onImageResize(uri);
    };

    onImageResize = uri => {
        MediaPicker.resizeImage(uri)
            .then(imageObj =>
                this.setState({
                    image_url: imageObj.uri,
                    isDpEdited: true
                })
            )
            .catch(err => console.log("err : ", err));
    };

    onImageAdd = ev => {
        if (this.state.editable) {
            MediaPicker.showPicker((uri, imageObj) =>
                this.onImagePicked(uri, imageObj)
            );
        }
    };

    onUpdateProfile = (api, payload, updateAlways) => {
        this.props.generalAction(
            api,
            payload,
            USER.SUCCESS,
            USER.FAILURE,
            user => this.onUpdateSuccess(user, updateAlways),
            this.onUpdateFailure
        );

        this.props.cbShowLoader(true);
    };

    onUpdateSuccess = (user, updateAlways) => {

        Utils.setUserLoggedIn(true);
        Utils.setUserToken(user.token);
        if (!updateAlways) {
            Utils.showMessage(
                "Success",
                "Profile updated successfully",
                "success"
            );
            this.props.cbShowLoader(false);
            setTimeout(() => {
                pop();
            }, 1000);
        }
    };

    onUpdateFailure = err => {
        this.props.cbShowLoader(false);
        Utils.showError(err);
    };

    avatar = () => {
        const { image_url } = this.state;

        return (
            <View style={styles.avatarStyle}>
                <Image
                   
                    source={Number.isInteger(image_url)?image_url:
                       
                    typeof image_url === "object" ?
                        { uri: image_url.uri }: { uri: IMAGE_BASE_URL+image_url }
                    
                    }
                    style={styles.imgStyle}
                    zIndex={999}
                />
                <TouchableOpacity
                    style={styles.iconStyle}
                    onPress={this.onImageAdd}
                >
                    <Image resizeMode="contain" source={icons.add_image} />
                </TouchableOpacity>
            </View>
        );
    };

    render() {
        const { user } = this.props;
        let userWithStatus =
            (user.data.user_type == "Contractor" &&
                user.data.is_personnel_file_status === 2) ||
            (user.data.user_type == "Facility" &&
                user.data.is_profile_verified === 2);
        return (
            <Fragment>
                <View style={styles.container}>
                    <View style={styles.topImageBackground} />
                    {userWithStatus && (
                        <Text
                            style={{
                                position: "absolute",
                                right: 0,
                                top: 0,
                                zIndex: 999,
                                margin: 20,
                                fontSize: 18,
                                color: "yellow"
                            }}
                        >
                            Verified
                        </Text>
                    )}
                    {this.avatar()}
                </View>
                <View style={styles.profileSec}>
                    {user.data.user_type === "Contractor" && (
                        <Contractor
                            data={user.data}
                            img={this.state.image_url}
                            editable={this.state.editable}
                            onUpdateProfile={this.onUpdateProfile}
                            isDpEdited={this.state.isDpEdited}
                            commentsList={this.state.commentsList}
                            {...this.props}
                        />
                    )}
                    {user.data.user_type === "Facility" && (
                        <Facility
                            data={user.data}
                            img={this.state.image_url}
                            editable={this.state.editable}
                            onUpdateProfile={this.onUpdateProfile}
                            isDpEdited={this.state.isDpEdited}
                            commentsList={this.state.commentsList}
                        />
                    )}
                </View>

                <Supporticon
                 clickBtn={() =>push("Support")}/>
            </Fragment>
        );
    }
}

const actions = { generalAction, generalSaveAction };

const mapStateToProps = state => ({
    user: state.user
});

const ComposeProfile = compose(
    connect(mapStateToProps, actions)(WithFetching(Profile))
);

export default ComposeProfile;

ComposeProfile.navigationOptions = ({ navigation }) => {
    const { state } = navigation;
    const params = state.params || {};
    return {
        title: "My Profile",
        headerStyle: styles.header,
        headerTitleStyle: styles.headerTitle,
        headerLeft: (
            <TouchableOpacity onPress={() => pop()}>
                <Image
                    style={{ margin: 10 }}
                    resizeMode="contain"
                    source={Images.backArrow}
                />
            </TouchableOpacity>
        ),
        // headerRight: (
        //     <View style={{ flexDirection: "row" }}>
        //         <TouchableOpacity onPress={params.onEdit}>
        //             <Image
        //                 style={{ margin: 10 }}
        //                 resizeMode="contain"
        //                 source={icons.ic_edit}
        //             />
        //         </TouchableOpacity>
        //     </View>
        // )
    };
};
