import React, { Component } from "react";
import { View, Text, StyleSheet, TouchableOpacity, Image } from "react-native";
import { connect } from "react-redux";
import { FlatListHandler, MainCard } from "../../components";
import {
    API_GET_CONTRACTS,
    API_FACILITY_INVITATION
} from "../../services/WebService";
import { pop } from "../../services/NavigationService";
import { colors, UIFont, fonts, icons } from "../../assets";

import { generalAction, notificationBar } from "../../actions/GeneralActions";
import {
    LIST_ACTIVE_CONTRACTS,
    SHOW_BAR,
    HIDE_BAR,
    DUMP
} from "../../actions/ActionTypes";

class SelectShiftContract extends Component {
    static navigationOptions = ({ navigation }) => {
        return {
            headerStyle: styles.header,
            headerTitleStyle: styles.headerTitle,
            headerLeft: (
                <TouchableOpacity onPress={() => pop()}>
                    <Image
                        style={{ margin: 10 }}
                        resizeMode="contain"
                        source={icons.ic_arrow_backward_black}
                    />
                </TouchableOpacity>
            ),
            headerRight: (
                <View
                    style={{
                        backgroundColor: colors.HeaderColor,
                        width: "10%"
                    }}
                />
            )
        };
    };
    constructor(props) {
        super(props);
        this.state = {
            isFetching: false
        };
    }

    componentDidMount() {
        this.fetchData();
    }

    fetchData = (isConcat: boolean = false, page: Number = 1) => {
        this.setState({ isFetching: true });
        this.props.generalAction(
            API_GET_CONTRACTS,
            {
                user_id: this.props.user.id,
                contract_status: "active",
                is_contract_complete: "incomplete",
                is_onboarded: 0,
                category_id: this.props.navigation.getParam("category_id"),
                page
            },
            LIST_ACTIVE_CONTRACTS.SUCCESS,
            LIST_ACTIVE_CONTRACTS.FAILURE,
            () => {
                this.setState({ isFetching: false });
            },
            () => {
                this.setState({ isFetching: false });
            },
            true,
            isConcat
        );
    };

    goTo = data => {
        const { navigation } = this.props;
        let payload = new FormData();
        payload.append("user_id", navigation.getParam("id"));
        payload.append("contract_id", data.contract_id);
        payload.append("invitation_status", "pending");
        payload.append("invitation_type", "sent");
        this.props.generalAction(
            API_FACILITY_INVITATION,
            payload,
            DUMP.SUCCESS,
            DUMP.FAILURE,
            () => {
                pop(2);
                this.invitationSent(data);
            },
            () => {
                this.goTo(data);
            }
        );
    };

    invitationSent({ image_url, full_name }) {
        let obj = {
            image_url,
            full_name,
            message: "Your request has been sent"
        };
        this.props.notificationBar(SHOW_BAR, obj);
        setTimeout(() => {
            this.props.notificationBar(HIDE_BAR);
        }, 3000);
    }

    renderItem = ({ item, index }, onDelete, goTo) => {
        let routeName = this.props.navigation.state.routeName;
        return (
            <MainCard
                item={item}
                onDelete={onDelete}
                goTo={data => this.goTo(data)}
                routeName={routeName}
            />
        );
    };

    render() {
        return (
            <View style={styles.containerActiveTab}>
                <View style={styles.heading}>
                    <Text style={styles.headingText}>
                        Select Contract/Shift
                    </Text>
                </View>
                <FlatListHandler
                    isFetching={this.state.isFetching}
                    refreshing={this.state.isFetching}
                    onRefresh={this.fetchData}
                    fetchRequest={this.fetchData}
                    data={this.props.activeContracts}
                    renderItem={item =>
                        this.renderItem(item, this.props.onDelete)
                    }
                />
            </View>
        );
    }
}

const actions = { generalAction, notificationBar };
const mapStateToProps = ({ user, activeContracts }) => ({
    user: user.data,
    activeContracts: activeContracts.data
});

export default connect(mapStateToProps, actions, null, { forwardRef: true })(
    SelectShiftContract
);

const styles = StyleSheet.create({
    containerActiveTab: {
        width: "100%",
        height: "100%",
        backgroundColor: colors.bgLightGrey
    },
    heading: {
        width: "100%",
        height: 40,
        paddingHorizontal: 15,
        paddingBottom: 5
    },
    headingText: {
        ...UIFont.font(fonts.semiBold, 16, colors.CardDate)
    },
    headerTitle: {
        color: colors.BlackColor,
        alignSelf: "center",
        textAlign: "center",
        justifyContent: "center",
        flex: 1,
        fontWeight: "bold",
        textAlignVertical: "center"
    },
    header: {
        borderBottomWidth: 0,
        backgroundColor: colors.HeaderColor,
        shadowColor: "transparent",
        elevation: 0,
        shadowRadius: 0,
        shadowOffset: {
            height: 0
        }
    }
});
